// Simple face detection using MediaPipe Face Detection (no TensorFlow.js needed)
// This works in Chrome extensions without unsafe-eval

console.log('=== SIMPLE TRACKER LOADED ===');

let video, canvas, ctx;
let isTracking = false;
let emotions = {
  stress: 0,
  excitement: 0,
  focus: 0,
  positive: 0
};

// Session tracking
let sessionStartTime = Date.now();
let emotionHistory = {
  stress: [],
  excitement: [],
  focus: [],
  positive: []
};
const MAX_HISTORY = 60;

// Face tracking variables
let lastFaceSize = 0;
let lastFaceX = 0;
let lastFaceY = 0;
let movementHistory = [];
const MOVEMENT_HISTORY_LENGTH = 30;

// Setup camera
async function setupCamera() {
  console.log('=== SETUP CAMERA ===');
  video = document.getElementById('video');
  canvas = document.getElementById('canvas');
  
  if (!video || !canvas) {
    console.error('Video or canvas element not found');
    return false;
  }
  
  ctx = canvas.getContext('2d');
  
  try {
    const stream = await navigator.mediaDevices.getUserMedia({
      video: { width: 640, height: 480, facingMode: 'user' }
    });
    
    video.srcObject = stream;
    
    return new Promise((resolve) => {
      video.onloadedmetadata = () => {
        video.play();
        canvas.width = video.videoWidth;
        canvas.height = video.videoHeight;
        console.log('Camera ready:', canvas.width, 'x', canvas.height);
        resolve(true);
      };
    });
  } catch (error) {
    console.error('Camera error:', error);
    return false;
  }
}

// Simple face detection using pixel analysis
function detectFaceSimple() {
  if (!isTracking || !video || !ctx) {
    requestAnimationFrame(detectFaceSimple);
    return;
  }
  
  try {
    // Draw video frame to canvas
    ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
    
    // Get image data
    const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
    const data = imageData.data;
    
    // Simple skin tone detection (very basic face detection)
    let facePixels = [];
    let totalSkinPixels = 0;
    let minX = canvas.width, maxX = 0, minY = canvas.height, maxY = 0;
    
    // Scan for skin-colored pixels
    for (let y = 0; y < canvas.height; y += 4) {
      for (let x = 0; x < canvas.width; x += 4) {
        const i = (y * canvas.width + x) * 4;
        const r = data[i];
        const g = data[i + 1];
        const b = data[i + 2];
        
        // Simple skin tone detection
        if (isSkinTone(r, g, b)) {
          totalSkinPixels++;
          if (x < minX) minX = x;
          if (x > maxX) maxX = x;
          if (y < minY) minY = y;
          if (y > maxY) maxY = y;
        }
      }
    }
    
    // If we found enough skin pixels, consider it a face
    if (totalSkinPixels > 500) {
      const faceWidth = maxX - minX;
      const faceHeight = maxY - minY;
      const faceCenterX = (minX + maxX) / 2;
      const faceCenterY = (minY + maxY) / 2;
      
      // Draw green box around detected face
      ctx.strokeStyle = '#00ff88';
      ctx.lineWidth = 3;
      ctx.strokeRect(minX, minY, faceWidth, faceHeight);
      
      // Draw center dot
      ctx.fillStyle = '#00bbff';
      ctx.beginPath();
      ctx.arc(faceCenterX, faceCenterY, 5, 0, 2 * Math.PI);
      ctx.fill();
      
      // Calculate emotions based on face metrics
      calculateEmotions(faceWidth, faceHeight, faceCenterX, faceCenterY);
      
      console.log('Face detected:', faceWidth, 'x', faceHeight);
    } else {
      // No face detected - decay emotions
      emotions.stress *= 0.95;
      emotions.excitement *= 0.95;
      emotions.focus *= 0.95;
      emotions.positive *= 0.95;
    }
    
    // Update emotion history
    updateEmotionHistory();
    
    // Save to storage for popup
    chrome.storage.local.set({ emotions, sessionStart: sessionStartTime });
    
  } catch (error) {
    console.error('Detection error:', error);
  }
  
  requestAnimationFrame(detectFaceSimple);
}

// Check if RGB values match skin tone
function isSkinTone(r, g, b) {
  // Simple skin tone detection algorithm
  return (
    r > 95 && g > 40 && b > 20 &&
    r > g && r > b &&
    Math.abs(r - g) > 15 &&
    Math.max(r, g, b) - Math.min(r, g, b) > 15
  );
}

// Calculate emotions from face metrics
function calculateEmotions(faceWidth, faceHeight, faceCenterX, faceCenterY) {
  const frameWidth = canvas.width;
  const frameHeight = canvas.height;
  
  // Calculate face size ratio (leaning in = excitement)
  const faceSize = (faceWidth * faceHeight) / (frameWidth * frameHeight);
  const excitement = Math.min(90, Math.max(10, (faceSize - 0.05) * 300));
  
  // Calculate movement (jittery = stress)
  const movement = Math.sqrt(
    Math.pow(faceCenterX - lastFaceX, 2) + 
    Math.pow(faceCenterY - lastFaceY, 2)
  );
  
  movementHistory.push(movement);
  if (movementHistory.length > MOVEMENT_HISTORY_LENGTH) {
    movementHistory.shift();
  }
  
  const avgMovement = movementHistory.reduce((a, b) => a + b, 0) / movementHistory.length;
  const stress = Math.min(90, Math.max(10, avgMovement * 2));
  
  // Calculate focus (centered and stable = focused)
  const distanceFromCenter = Math.sqrt(
    Math.pow(faceCenterX - frameWidth / 2, 2) + 
    Math.pow(faceCenterY - frameHeight / 2, 2)
  );
  const maxDistance = Math.sqrt(Math.pow(frameWidth / 2, 2) + Math.pow(frameHeight / 2, 2));
  const centeredness = 1 - (distanceFromCenter / maxDistance);
  const focus = Math.min(90, Math.max(10, centeredness * 100 - avgMovement));
  
  // Positive baseline
  const positive = 50 + (focus - stress) * 0.3;
  
  // Smooth the values
  emotions.stress = smooth(emotions.stress, stress, 0.3);
  emotions.excitement = smooth(emotions.excitement, excitement, 0.3);
  emotions.focus = smooth(emotions.focus, focus, 0.3);
  emotions.positive = smooth(emotions.positive, positive, 0.3);
  
  // Update last position
  lastFaceX = faceCenterX;
  lastFaceY = faceCenterY;
  lastFaceSize = faceSize;
}

// Smooth value changes
function smooth(current, target, factor) {
  return current + (target - current) * factor;
}

// Update emotion history
function updateEmotionHistory() {
  emotionHistory.stress.push(emotions.stress);
  emotionHistory.excitement.push(emotions.excitement);
  emotionHistory.focus.push(emotions.focus);
  emotionHistory.positive.push(emotions.positive);
  
  if (emotionHistory.stress.length > MAX_HISTORY) {
    emotionHistory.stress.shift();
    emotionHistory.excitement.shift();
    emotionHistory.focus.shift();
    emotionHistory.positive.shift();
  }
}

// Get current emotions
function getEmotions() {
  return { ...emotions };
}

// Get emotion history
function getEmotionHistory() {
  return { ...emotionHistory };
}

// Start tracking
async function startTracking() {
  if (isTracking) return true;
  
  console.log('=== STARTING SIMPLE TRACKER ===');
  
  const cameraReady = await setupCamera();
  if (!cameraReady) {
    console.error('Failed to setup camera');
    return false;
  }
  
  isTracking = true;
  sessionStartTime = Date.now();
  detectFaceSimple();
  
  console.log('=== SIMPLE TRACKER STARTED ===');
  return true;
}

// Stop tracking
function stopTracking() {
  isTracking = false;
  if (video && video.srcObject) {
    const tracks = video.srcObject.getTracks();
    tracks.forEach(track => track.stop());
  }
}

// Export functions
window.EmotionTracker = {
  start: startTracking,
  stop: stopTracking,
  getEmotions,
  getEmotionHistory,
  isTracking: () => isTracking
};

console.log('Simple tracker ready');
