// Simple fake emotions - just animate the bars with realistic data
// No face detection, no complex tracking, just visual demo

console.log('=== SIMPLE FAKE EMOTIONS LOADED ===');

let emotions = { stress: 35, excitement: 45, focus: 65, positive: 55 };

const scenarios = [
  { stress: 20, excitement: 25, focus: 75, positive: 55, duration: 8000 },
  { stress: 30, excitement: 20, focus: 85, positive: 50, duration: 10000 },
  { stress: 35, excitement: 65, focus: 60, positive: 70, duration: 6000 },
  { stress: 70, excitement: 55, focus: 40, positive: 30, duration: 5000 },
  { stress: 45, excitement: 40, focus: 55, positive: 45, duration: 7000 }
];

let currentScenario = 0;
let scenarioStartTime = Date.now();

// Session statistics tracking
let sessionStats = {
  totalStress: 0,
  totalExcitement: 0,
  totalFocus: 0,
  peakStress: 0,
  peakExcitement: 0,
  timeInOptimal: 0,
  timeInCaution: 0,
  timeInDanger: 0,
  dataPoints: 0
};

// Zone tracking and alerts
let lastZone = 'optimal';
let alerts = [];

function updateEmotions() {
  const now = Date.now();
  const elapsed = now - scenarioStartTime;
  
  // Switch scenario
  if (elapsed > scenarios[currentScenario].duration) {
    currentScenario = (currentScenario + 1) % scenarios.length;
    scenarioStartTime = now;
    console.log('Scenario changed:', currentScenario);
  }
  
  const target = scenarios[currentScenario];
  
  // Smooth transition
  emotions.stress += (target.stress - emotions.stress) * 0.02;
  emotions.excitement += (target.excitement - emotions.excitement) * 0.02;
  emotions.focus += (target.focus - emotions.focus) * 0.02;
  emotions.positive += (target.positive - emotions.positive) * 0.02;
  
  // Random fluctuation
  emotions.stress += (Math.random() - 0.5) * 2;
  emotions.excitement += (Math.random() - 0.5) * 2;
  emotions.focus += (Math.random() - 0.5) * 1.5;
  emotions.positive += (Math.random() - 0.5) * 1.5;
  
  // Clamp
  emotions.stress = Math.max(10, Math.min(95, emotions.stress));
  emotions.excitement = Math.max(10, Math.min(95, emotions.excitement));
  emotions.focus = Math.max(10, Math.min(95, emotions.focus));
  emotions.positive = Math.max(10, Math.min(95, emotions.positive));
  
  // Update UI directly
  updateBars();
  
  // Update session stats (every 100ms)
  updateSessionStats();
  
  // Check for zone changes (for alerts)
  checkZoneChange();
}

function updateBars() {
  // Update stress
  const stressBar = document.getElementById('stress-bar');
  const stressValue = document.getElementById('stress-value');
  if (stressBar && stressValue) {
    stressBar.style.width = Math.round(emotions.stress) + '%';
    stressValue.textContent = Math.round(emotions.stress) + '%';
  }
  
  // Update excitement
  const excitementBar = document.getElementById('excitement-bar');
  const excitementValue = document.getElementById('excitement-value');
  if (excitementBar && excitementValue) {
    excitementBar.style.width = Math.round(emotions.excitement) + '%';
    excitementValue.textContent = Math.round(emotions.excitement) + '%';
  }
  
  // Update focus
  const focusBar = document.getElementById('focus-bar');
  const focusValue = document.getElementById('focus-value');
  if (focusBar && focusValue) {
    focusBar.style.width = Math.round(emotions.focus) + '%';
    focusValue.textContent = Math.round(emotions.focus) + '%';
  }
  
  // Update positive
  const positiveBar = document.getElementById('positive-bar');
  const positiveValue = document.getElementById('positive-value');
  if (positiveBar && positiveValue) {
    positiveBar.style.width = Math.round(emotions.positive) + '%';
    positiveValue.textContent = Math.round(emotions.positive) + '%';
  }
}

// Update session statistics
function updateSessionStats() {
  sessionStats.dataPoints++;
  
  sessionStats.totalStress += emotions.stress;
  sessionStats.totalExcitement += emotions.excitement;
  sessionStats.totalFocus += emotions.focus;
  
  sessionStats.peakStress = Math.max(sessionStats.peakStress, emotions.stress);
  sessionStats.peakExcitement = Math.max(sessionStats.peakExcitement, emotions.excitement);
  
  const s = emotions.stress;
  const e = emotions.excitement;
  const f = emotions.focus;
  
  if (s > 65 || e > 70 || f < 35) {
    sessionStats.timeInDanger++;
  } else if (s > 50 || e > 55 || f < 50) {
    sessionStats.timeInCaution++;
  } else {
    sessionStats.timeInOptimal++;
  }
  
  updateStatsUI();
}

// Update stats UI
function updateStatsUI() {
  if (sessionStats.dataPoints === 0) return;
  
  const avgStress = Math.round(sessionStats.totalStress / sessionStats.dataPoints);
  const avgExcitement = Math.round(sessionStats.totalExcitement / sessionStats.dataPoints);
  const avgFocus = Math.round(sessionStats.totalFocus / sessionStats.dataPoints);
  
  const totalTime = sessionStats.timeInOptimal + sessionStats.timeInCaution + sessionStats.timeInDanger;
  const optimalPercent = totalTime > 0 ? Math.round((sessionStats.timeInOptimal / totalTime) * 100) : 0;
  
  const stats = document.querySelectorAll('.stat-value');
  if (stats.length >= 6) {
    stats[0].textContent = avgStress + '%';
    stats[1].textContent = avgExcitement + '%';
    stats[2].textContent = avgFocus + '%';
    stats[3].textContent = Math.round(sessionStats.peakStress) + '%';
    stats[4].textContent = Math.round(sessionStats.peakExcitement) + '%';
    stats[5].textContent = optimalPercent + '%';
  }
}

// Check for zone changes and add alerts
function checkZoneChange() {
  const s = emotions.stress;
  const e = emotions.excitement;
  const f = emotions.focus;
  
  let currentZone;
  if (s > 65 || e > 70 || f < 35) {
    currentZone = 'danger';
  } else if (s > 50 || e > 55 || f < 50) {
    currentZone = 'caution';
  } else {
    currentZone = 'optimal';
  }
  
  if (currentZone !== lastZone) {
    const time = new Date().toLocaleTimeString();
    let message = '';
    
    if (currentZone === 'danger') {
      message = `Entered DANGER zone - Stress: ${Math.round(s)}%, Excitement: ${Math.round(e)}%, Focus: ${Math.round(f)}%`;
    } else if (currentZone === 'caution') {
      message = `Entered CAUTION zone`;
    } else {
      message = `Returned to OPTIMAL zone`;
    }
    
    addAlert(time, message);
    lastZone = currentZone;
  }
}

// Add alert to history
function addAlert(time, message) {
  alerts.unshift({ time, message });
  
  if (alerts.length > 10) {
    alerts = alerts.slice(0, 10);
  }
  
  updateAlertsUI();
}

// Update alerts UI
function updateAlertsUI() {
  const alertsList = document.getElementById('alerts-list');
  if (!alertsList) return;
  
  alertsList.innerHTML = '';
  
  if (alerts.length === 0) {
    const alertItem = document.createElement('div');
    alertItem.className = 'alert-item';
    alertItem.textContent = 'Session started';
    alertsList.appendChild(alertItem);
  } else {
    alerts.forEach(alert => {
      const alertItem = document.createElement('div');
      alertItem.className = 'alert-item';
      alertItem.textContent = `[${alert.time}] ${alert.message}`;
      alertsList.appendChild(alertItem);
    });
  }
}

// Setup camera feed
async function setupCamera() {
  const video = document.getElementById('video');
  const canvas = document.getElementById('canvas');
  
  if (!video || !canvas) {
    console.log('Video/canvas elements not found - skipping camera');
    return;
  }
  
  try {
    const stream = await navigator.mediaDevices.getUserMedia({
      video: { 
        width: 640, 
        height: 480,
        facingMode: 'user'
      }
    });
    
    video.srcObject = stream;
    
    return new Promise((resolve) => {
      video.onloadedmetadata = () => {
        video.play();
        canvas.width = video.videoWidth;
        canvas.height = video.videoHeight;
        console.log('Camera initialized:', video.videoWidth, 'x', video.videoHeight);
        
        // Start fake tracking animation
        drawFakeTracking();
        
        resolve();
      };
    });
  } catch (error) {
    console.error('Camera access error:', error);
  }
}

// Draw fake tracking box on canvas
function drawFakeTracking() {
  const canvas = document.getElementById('canvas');
  const ctx = canvas?.getContext('2d');
  
  if (!ctx) return;
  
  const time = Date.now() / 1000;
  const centerX = canvas.width / 2;
  const centerY = canvas.height / 2.5;
  const boxWidth = canvas.width * 0.35;
  const boxHeight = canvas.height * 0.45;
  
  // Animate position slightly
  const offsetX = Math.sin(time * 0.5) * 15;
  const offsetY = Math.cos(time * 0.4) * 10;
  
  ctx.clearRect(0, 0, canvas.width, canvas.height);
  
  // Draw green tracking box
  ctx.strokeStyle = '#00ff88';
  ctx.lineWidth = 3;
  ctx.strokeRect(
    centerX - boxWidth/2 + offsetX,
    centerY - boxHeight/2 + offsetY,
    boxWidth,
    boxHeight
  );
  
  // Draw "TRACKING..." label
  ctx.fillStyle = '#00ff88';
  ctx.font = 'bold 12px Courier New';
  ctx.fillText('TRACKING...', centerX - boxWidth/2 + offsetX, centerY - boxHeight/2 + offsetY - 10);
  
  requestAnimationFrame(drawFakeTracking);
}

// Start when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}

async function init() {
  console.log('Starting simple fake emotions...');
  
  // Setup camera
  await setupCamera();
  
  // Force first update
  setTimeout(() => {
    updateBars();
    console.log('First update done');
  }, 500);
  
  // Add initial alert
  setTimeout(() => {
    addAlert(new Date().toLocaleTimeString(), 'System initialized. Tracking started.');
  }, 1000);
  
  // Update every 100ms
  setInterval(updateEmotions, 100);
  
  console.log('Fake emotions running');
}
