// Popup.js - Mini view controller with fake emotions

let sessionStartTime = Date.now();
let emotions = {
  stress: 35,
  excitement: 45,
  focus: 65,
  positive: 55
};

const scenarios = [
  { stress: 20, excitement: 25, focus: 75, positive: 55, duration: 8000 },
  { stress: 30, excitement: 20, focus: 85, positive: 50, duration: 10000 },
  { stress: 35, excitement: 65, focus: 60, positive: 70, duration: 6000 },
  { stress: 70, excitement: 55, focus: 40, positive: 30, duration: 5000 },
  { stress: 45, excitement: 40, focus: 55, positive: 45, duration: 7000 }
];

let currentScenario = 0;
let scenarioStartTime = Date.now();

// Initialize
document.addEventListener('DOMContentLoaded', () => {
  console.log('Popup initializing...');
  
  // Initialize mini camera
  initMiniCamera();
  
  // Update session time
  setInterval(updateSessionTime, 1000);

  // Open full tracker button
  document.getElementById('open-full-btn').addEventListener('click', () => {
    chrome.tabs.create({ url: 'fullpage.html' });
  });

  // Start emotion updates
  setInterval(updatePopupEmotions, 100);
  
  // Force first update
  setTimeout(() => {
    updateUI();
    console.log('Popup first update done');
  }, 500);
});

function updateUI() {
  // Update emotion bars
  updateBar('stress', emotions.stress);
  updateBar('excitement', emotions.excitement);
  updateBar('focus', emotions.focus);
  updateBar('positive', emotions.positive);

  // Update trading zone
  updateTradingZone();

  // Update emotional state
  updateEmotionalState();
}

function updateBar(emotion, value) {
  const bar = document.getElementById(`${emotion}-bar`);
  const valueLabel = document.getElementById(`${emotion}-value`);
  
  if (bar && valueLabel) {
    bar.style.width = `${value}%`;
    valueLabel.textContent = `${Math.round(value)}%`;
  }
}

function updateTradingZone() {
  const zoneIndicator = document.getElementById('zone-indicator');
  const zoneIcon = zoneIndicator.querySelector('.zone-icon');
  const zoneText = zoneIndicator.querySelector('.zone-text');
  const { stress, excitement, focus } = emotions;

  // Determine zone based on emotion levels
  if (stress > 70 || excitement > 70 || focus < 25) {
    zoneIndicator.className = 'zone-indicator danger';
    zoneIcon.textContent = '🚫';
    zoneText.textContent = 'DANGER';
  } else if (stress > 45 || excitement > 45 || focus < 40) {
    zoneIndicator.className = 'zone-indicator caution';
    zoneIcon.textContent = '⚠️';
    zoneText.textContent = 'CAUTION';
  } else {
    zoneIndicator.className = 'zone-indicator optimal';
    zoneIcon.textContent = '✓';
    zoneText.textContent = 'OPTIMAL';
  }
}

function updateEmotionalState() {
  const stateLabel = document.getElementById('emotional-state');
  const { stress, focus } = emotions;

  if (stress > 60 && focus > 50) {
    stateLabel.textContent = 'FOCUSED';
    stateLabel.style.color = 'var(--accent-green)';
  } else if (stress > 60) {
    stateLabel.textContent = 'STRESSED';
    stateLabel.style.color = 'var(--danger)';
  } else {
    stateLabel.textContent = 'NEUTRAL';
    stateLabel.style.color = 'var(--text-secondary)';
  }
}

function updateSessionTime() {
  const elapsed = Date.now() - sessionStartTime;
  const minutes = Math.floor(elapsed / 60000);
  const seconds = Math.floor((elapsed % 60000) / 1000);
  
  const timeLabel = document.getElementById('session-time');
  timeLabel.textContent = `${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`;
}

// Update emotions based on scenarios
function updatePopupEmotions() {
  const now = Date.now();
  const elapsed = now - scenarioStartTime;
  
  // Switch scenario
  if (elapsed > scenarios[currentScenario].duration) {
    currentScenario = (currentScenario + 1) % scenarios.length;
    scenarioStartTime = now;
  }
  
  const target = scenarios[currentScenario];
  
  // Smooth transition
  emotions.stress += (target.stress - emotions.stress) * 0.02;
  emotions.excitement += (target.excitement - emotions.excitement) * 0.02;
  emotions.focus += (target.focus - emotions.focus) * 0.02;
  emotions.positive += (target.positive - emotions.positive) * 0.02;
  
  // Random fluctuation
  emotions.stress += (Math.random() - 0.5) * 2;
  emotions.excitement += (Math.random() - 0.5) * 2;
  emotions.focus += (Math.random() - 0.5) * 1.5;
  emotions.positive += (Math.random() - 0.5) * 1.5;
  
  // Clamp
  emotions.stress = Math.max(10, Math.min(95, emotions.stress));
  emotions.excitement = Math.max(10, Math.min(95, emotions.excitement));
  emotions.focus = Math.max(10, Math.min(95, emotions.focus));
  emotions.positive = Math.max(10, Math.min(95, emotions.positive));
  
  // Update UI
  updateUI();
}

// Initialize mini camera in popup
async function initMiniCamera() {
  const video = document.getElementById('popup-video');
  
  if (!video) {
    console.log('Popup video element not found');
    return;
  }
  
  try {
    const stream = await navigator.mediaDevices.getUserMedia({
      video: {
        width: 320,
        height: 240,
        facingMode: 'user'
      }
    });
    
    video.srcObject = stream;
    console.log('Popup camera initialized');
  } catch (error) {
    console.error('Popup camera error:', error);
    // Hide camera section if access denied
    const cameraSection = document.querySelector('.mini-camera-section');
    if (cameraSection) {
      cameraSection.style.display = 'none';
    }
  }
}
